"""Custom decorators for access control."""
from __future__ import annotations

from functools import wraps
from typing import Callable, TypeVar

from flask import abort
from flask_login import current_user, login_required

F = TypeVar("F", bound=Callable[..., object])


def role_required(*roles: str) -> Callable[[F], F]:
    """Ensure the current user has one of the specified roles."""

    def decorator(func: F) -> F:
        @wraps(func)
        @login_required
        def wrapped(*args, **kwargs):  # type: ignore[override]
            if current_user.role not in roles:
                abort(403)
            return func(*args, **kwargs)

        return wrapped  # type: ignore[return-value]

    return decorator


def admin_required(func: F) -> F:
    """Shortcut for requiring administrator access."""

    @wraps(func)
    @login_required
    def wrapped(*args, **kwargs):  # type: ignore[override]
        if not current_user.is_admin:
            abort(403)
        return func(*args, **kwargs)

    return wrapped  # type: ignore[return-value]
